#!/usr/bin/env python3
import sys
import os
import re
import logging
import datetime
import asyncio

from PyQt6.QtCore import (
    Qt, QObject, pyqtSignal, QRunnable, pyqtSlot, QThreadPool, QStandardPaths,
    QPoint, QTimer, QSettings, QLocale, QTranslator
)
from PyQt6.QtGui import QFont, QCursor, QIntValidator
from PyQt6.QtWidgets import (
    QApplication, QMainWindow, QWidget, QVBoxLayout, QHBoxLayout,
    QLabel, QMessageBox, QScrollArea, QTabWidget, QProgressBar,
    QFrame, QGraphicsDropShadowEffect
)

try:
    from qt_material import apply_stylesheet
    HAS_MATERIAL = True
except ImportError:
    HAS_MATERIAL = False

from .models import Anime, STYLES
from .database import get_database_connection, add_to_list, remove_from_list, get_list
from .api import AsyncSearchWorker, AsyncWorkerThread, get_popular_anime
from .utils import load_pixmap, WorkerSignals
from .ui_components import MaterialCard, RoundButton, MaterialLineEdit, MaterialComboBox

# Version
__version__ = "0.0.5"

# Logging konfigurieren
logging.basicConfig(
    filename="shokaishelf.log",
    level=logging.INFO,
    format="%(asctime)s %(levelname)s: %(message)s"
)
logger = logging.getLogger(__name__)

# QSettings für Persistenz
settings = QSettings("Chaptiv", "ShokaiShelf")

# Worker für Top-3-Animes
class PopularAnimeWorker(QRunnable):
    def __init__(self, year, per_page, async_loop):
        super().__init__()
        self.year = year
        self.per_page = per_page
        self.async_loop = async_loop
        self.signals = WorkerSignals()

    @pyqtSlot()
    def run(self):
        coro = self._fetch()
        future = asyncio.run_coroutine_threadsafe(coro, self.async_loop)
        future.add_done_callback(lambda f: self._handle_result(f))

    async def _fetch(self):
        return await get_popular_anime(self.year, self.per_page)

    def _handle_result(self, future):
        try:
            results = future.result()
            self.signals.finished.emit(results)
        except Exception as e:
            self.signals.error.emit(str(e))

# Hauptfenster
class AnimeManager(QMainWindow):
    def __init__(self):
        super().__init__()
        logger.info("Initializing AnimeManager")
        self.setWindowTitle("ShokaiShelf")
        self.resize(1000, 700)
        self.conn = get_database_connection()
        self.pool = QThreadPool.globalInstance()
        self.async_thread = AsyncWorkerThread()
        self.async_thread.start()
        self.use_appstore_layout = settings.value("use_appstore_layout", False, bool)

        central = QWidget()
        self.setCentralWidget(central)
        main_l = QVBoxLayout(central)
        main_l.setContentsMargins(16, 16, 16, 16)
        main_l.setSpacing(12)

        lbl = QLabel("ShokaiShelf")
        lbl.setFont(QFont("Arial", 18, QFont.Weight.Bold))
        lbl.setStyleSheet(f"color:{STYLES['primary']};")
        lbl.setAlignment(Qt.AlignmentFlag.AlignCenter)
        main_l.addWidget(lbl)

        self.tabs = QTabWidget()
        main_l.addWidget(self.tabs)
        self.search_tab = QWidget()
        self.watch_tab = QWidget()
        self.buy_tab = QWidget()
        self.tabs.addTab(self.search_tab, "Home")
        self.tabs.addTab(self.watch_tab, "Watch List")
        self.tabs.addTab(self.buy_tab, "Buy List")

        self.container_search = QWidget()
        self.results_search = QVBoxLayout(self.container_search)
        self._init_search_tab()

        self._init_list_tab(self.watch_tab, "watch_list")
        self._init_list_tab(self.buy_tab, "buy_list")

        logger.info("AnimeManager ready")

        # Zeige What's New-Popup, falls ein neues Update installiert wurde
        self.show_whats_new()

    def show_whats_new(self):
        """Zeigt das What's New-Popup basierend auf dem letzten Update."""
        last_version = settings.value("last_update_version", None)
        if last_version:
            key = f"seen_update_{last_version}"
            shown = settings.value(key, False, type=bool)
            if not shown:
                changelog = settings.value("last_update_changelog", "Keine Changelog-Informationen verfügbar.")
                QMessageBox.information(
                    self,
                    f"What’s New in {last_version}",
                    changelog
                )
                settings.setValue(key, True)

    def closeEvent(self, event):
        self.async_thread.stop()
        super().closeEvent(event)

    def _init_list_tab(self, tab, table):
        l = QVBoxLayout(tab)
        btn = RoundButton("Aktualisieren", primary=False)
        btn.clicked.connect(lambda: self.refresh_list(table))
        l.addWidget(btn, alignment=Qt.AlignmentFlag.AlignLeft)
        scroll = QScrollArea()
        scroll.setWidgetResizable(True)
        scroll.setStyleSheet(f"QScrollArea{{background:{STYLES['background']};border:none;}}")
        cont = QWidget()
        res = QVBoxLayout(cont)
        res.setContentsMargins(0, 0, 0, 0)
        res.setSpacing(8)
        scroll.setWidget(cont)
        l.addWidget(scroll, stretch=1)
        setattr(self, f"results_{table}", res)
        self.refresh_list(table)

    def refresh_list(self, table):
        logger.info(f"Refreshing {table}")
        layout = getattr(self, f"results_{table}")
        while layout.count():
            item = layout.takeAt(0)
            if item.widget():
                item.widget().deleteLater()
        entries = get_list(self.conn, table)
        if not entries:
            lbl = QLabel(f"{table.replace('_',' ').title()} is empty.")
            lbl.setStyleSheet(f"color:{STYLES['text_primary']}; border: none;")
            layout.addWidget(lbl)
        else:
            for anime in entries:
                card = self.create_anime_card(anime, table=table, show_buttons=True)
                layout.addWidget(card)
        layout.addStretch()

    def _init_search_tab(self):
        l = QVBoxLayout(self.search_tab)
        bar = QHBoxLayout()
        self.input_title = MaterialLineEdit("Anime-Titel eingeben...")
        self.input_title.returnPressed.connect(self.on_search)
        self.combo_genre = MaterialComboBox()
        self.combo_genre.addItem("Genre")
        self.combo_genre.addItems([
            "Action", "Adventure", "Comedy", "Drama", "Fantasy", "Horror",
            "Mystery", "Romance", "Sci-Fi", "Slice of Life", "Sports",
            "Supernatural", "Thriller", "Ecchi"
        ])
        self.input_year = MaterialLineEdit("Jahr (z.B. 2021)")
        self.input_year.setFixedWidth(120)
        current_year = datetime.datetime.now().year
        self.input_year.setPlaceholderText(f"Jahr (1900-{current_year})")
        self.input_year.setToolTip("Leer lassen für Suche ohne Jahresfilter")
        val = QIntValidator(1900, current_year, self.input_year)
        self.input_year.setValidator(val)
        self.input_year.textChanged.connect(self._validate_year_input)
        btn = RoundButton("Suchen")
        btn.clicked.connect(self.on_search)
        bar.addWidget(self.input_title, stretch=3)
        bar.addWidget(self.combo_genre, stretch=1)
        bar.addWidget(self.input_year)
        bar.addWidget(btn)
        l.addLayout(bar)
        toggle = RoundButton("Layout wechseln", primary=False)
        toggle.clicked.connect(self.toggle_layout)
        l.addWidget(toggle, alignment=Qt.AlignmentFlag.AlignRight)
        self.scroll_search = QScrollArea()
        self.scroll_search.setWidgetResizable(True)
        self.scroll_search.setWidget(self.container_search)
        l.addWidget(self.scroll_search, stretch=1)
        self._show_home_screen()

    def _validate_year_input(self, text):
        if not text or self.input_year.validator().validate(text, 0)[0] == QIntValidator.State.Acceptable:
            self.input_year.setStyleSheet(f"""
                QLineEdit {{ border:1px solid {STYLES['text_secondary']}; border-radius:9px;
                            padding:8px 12px; background-color:{STYLES['surface']}; color:{STYLES['text_primary']}; }}
                QLineEdit:focus {{ border:2px solid {STYLES['primary']}; }}
            """)
        else:
            self.input_year.setStyleSheet(f"""
                QLineEdit {{ border:2px solid {STYLES['error']}; border-radius:9px;
                            padding:8px 12px; background-color:{STYLES['surface']}; color:{STYLES['text_primary']}; }}
            """)

    def toggle_layout(self):
        self.use_appstore_layout = not self.use_appstore_layout
        settings.setValue("use_appstore_layout", self.use_appstore_layout)
        self._show_home_screen()
        self.refresh_list("watch_list")
        self.refresh_list("buy_list")

    def _show_home_screen(self):
        logger.info("Showing home screen")
        layout = self.results_search
        while layout.count():
            item = layout.takeAt(0)
            if item.widget():
                item.widget().deleteLater()
        lbl = QLabel("Willkommen bei ShokaiShelf!")
        lbl.setFont(QFont("Arial", 16, QFont.Weight.Bold))
        lbl.setAlignment(Qt.AlignmentFlag.AlignCenter)
        lbl.setStyleSheet(f"color:{STYLES['text_primary']}; border: none;")
        layout.addWidget(lbl)
        year = datetime.datetime.now().year
        worker = PopularAnimeWorker(year, per_page=3, async_loop=self.async_thread.loop)
        worker.signals.finished.connect(lambda res: self._on_popular_finished(res), Qt.ConnectionType.QueuedConnection)
        worker.signals.error.connect(lambda e: self._on_popular_error(e), Qt.ConnectionType.QueuedConnection)
        self.pool.start(worker)

    def _on_popular_finished(self, results):
        logger.info("Popular anime loaded")
        layout = self.results_search
        for anime in results:
            card = self.create_anime_card(anime)
            layout.addWidget(card)
        layout.addStretch()
        layout.update()

    def _on_popular_error(self, msg):
        logger.error(f"Popular anime error: {msg}")
        layout = self.results_search
        err = QLabel("Konnte beliebteste Anime nicht laden.")
        err.setAlignment(Qt.AlignmentFlag.AlignCenter)
        err.setStyleSheet(f"color:{STYLES['error']}; border: none;")
        layout.addWidget(err)
        layout.addStretch()
        layout.update()

    def on_search(self):
        q = self.input_title.text().strip()
        if not q:
            QMessageBox.warning(self, "Eingabe fehlt", "Bitte einen Titel eingeben.")
            return
        genre = self.combo_genre.currentText()
        genre = None if genre == "Genre" else genre
        year = self.input_year.text()
        year = int(year) if year.isdigit() else None
        layout = self.results_search
        while layout.count():
            item = layout.takeAt(0)
            if item.widget():
                item.widget().deleteLater()
        loading = QLabel("Lade…")
        loading.setStyleSheet(f"color:{STYLES['text_primary']}; border: none;")
        layout.addWidget(loading)
        worker = AsyncSearchWorker(q, genre, year, self.async_thread.loop)
        worker.signals.finished.connect(lambda res: self._on_search_finished(res, loading), Qt.ConnectionType.QueuedConnection)
        worker.signals.error.connect(lambda e: self._on_search_error(e, loading), Qt.ConnectionType.QueuedConnection)
        self.pool.start(worker)

    def _on_search_finished(self, results, loading):
        logger.info("Search finished")
        loading.deleteLater()
        layout = self.results_search
        if not results:
            no = QLabel("Keine Treffer.")
            no.setStyleSheet(f"color:{STYLES['text_primary']}; border: none;")
            layout.addWidget(no)
        else:
            for anime in results:
                card = self.create_anime_card(anime)
                layout.addWidget(card)
        layout.addStretch()

    def _on_search_error(self, msg, loading):
        logger.error(f"Search error: {msg}")
        loading.deleteLater()
        user_msg = "Ein unbekannter Fehler ist aufgetreten."
        if "ConnectionError" in msg:
            user_msg = "Keine Internetverbindung. Bitte überprüfen Sie Ihre Netzwerkverbindung."
        elif "HTTPError" in msg:
            user_msg = "Der Server hat einen Fehler zurückgegeben. Bitte versuchen Sie es später erneut."
        dlg = QMessageBox(self)
        dlg.setIcon(QMessageBox.Icon.Critical)
        dlg.setWindowTitle("Netzwerkfehler")
        dlg.setText(user_msg)
        dlg.setDetailedText(msg)
        dlg.setStandardButtons(QMessageBox.StandardButton.Retry | QMessageBox.StandardButton.Cancel)
        if dlg.exec() == QMessageBox.StandardButton.Retry:
            self.on_search()

    def _add_watch(self, anime: Anime):
        logger.info(f"Attempting to add to watch_list: {anime.title}")
        watch_list = get_list(self.conn, "watch_list")
        if any(entry.title == anime.title for entry in watch_list):
            QMessageBox.information(self, "Bereits in Liste", f"{anime.title} ist bereits in der Watch List.")
            return
        add_to_list(self.conn, "watch_list", anime)
        QMessageBox.information(self, "Erfolg", f"{anime.title} zur Watch List hinzugefügt.")
        self.refresh_list("watch_list")

    def _add_buy(self, anime: Anime):
        logger.info(f"Attempting to add to buy_list: {anime.title}")
        buy_list = get_list(self.conn, "buy_list")
        if any(entry.title == anime.title for entry in buy_list):
            QMessageBox.information(self, "Bereits in Liste", f"{anime.title} ist bereits in der Buy List.")
            return
        add_to_list(self.conn, "buy_list", anime)
        QMessageBox.information(self, "Erfolg", f"{anime.title} zur Buy List hinzugefügt.")
        self.refresh_list("buy_list")

    def create_anime_card(self, anime: Anime, table: str = None, show_buttons: bool = True):
        card = MaterialCard(appstore_layout=self.use_appstore_layout)
        title = QLabel(anime.title, card)
        title.setFont(QFont("Arial", 14, QFont.Weight.Bold))
        title.setWordWrap(True)
        title.setStyleSheet(f"color:{STYLES['text_primary']}; border: none; background: transparent;")

        img_size = (200, 300) if self.use_appstore_layout else (150, 200)
        pix = load_pixmap(anime.cover_url, img_size, lambda u, p: img.setPixmap(p))
        img = QLabel(card)
        img.setPixmap(pix)
        img.setFixedSize(*img_size)
        img.setAlignment(Qt.AlignmentFlag.AlignCenter)

        desc = re.sub(r'<[^>]+>', '', anime.description)
        if not desc:
            desc = "Keine Beschreibung verfügbar."
        d_lbl = QLabel(desc, card)
        d_lbl.setWordWrap(True)
        d_lbl.setStyleSheet(f"color:{STYLES['text_primary']}; border: none; background: transparent;")
        d_lbl.setMaximumHeight(100 if self.use_appstore_layout else 120)

        g_lbl = QLabel("Genres: " + ", ".join(anime.genres), card)
        s_lbl = QLabel(f"Score: {anime.score or 'N/A'}", card)
        y_lbl = QLabel(f"Jahr: {anime.year or 'N/A'}", card)
        e_lbl = QLabel(f"Episoden: {anime.episodes or 'N/A'}", card)
        for lbl in (g_lbl, s_lbl, y_lbl, e_lbl):
            lbl.setStyleSheet(f"color:{STYLES['text_secondary']}; border: none; background: transparent;")

        info = QVBoxLayout()
        info.setSpacing(8)
        info.addWidget(title)
        info.addWidget(d_lbl)
        info.addWidget(g_lbl)
        info.addWidget(s_lbl)
        info.addWidget(y_lbl)
        info.addWidget(e_lbl)

        if show_buttons:
            if table:
                btn_r = RoundButton("Entfernen", primary=False)
                btn_r.clicked.connect(lambda: (remove_from_list(self.conn, table, anime.title), self.refresh_list(table)))
                info.addWidget(btn_r, alignment=Qt.AlignmentFlag.AlignRight)
            else:
                btn_w = RoundButton("Zur Watch List")
                btn_w.clicked.connect(lambda: self._add_watch(anime))
                btn_b = RoundButton("Zur Buy List")
                btn_b.clicked.connect(lambda: self._add_buy(anime))
                btns = QHBoxLayout()
                if not self.use_appstore_layout:
                    btns.addStretch()
                btns.addWidget(btn_w)
                btns.addWidget(btn_b)
                info.addLayout(btns)

        card.layout.addWidget(img)
        card.layout.addLayout(info, stretch=1 if not self.use_appstore_layout else 0)
        if self.use_appstore_layout:
            card.layout.addStretch()
        return card

# SplashScreen mit Self-Tests
class InitSignals(QObject):
    progress = pyqtSignal(int)

class InitWorker(QRunnable):
    def __init__(self, fn, args, weight, signals: InitSignals):
        super().__init__()
        self.fn = fn
        self.args = args
        self.weight = weight
        self.signals = signals

    @pyqtSlot()
    def run(self):
        try:
            self.fn(*self.args)
            if self.weight > 0:
                self.signals.progress.emit(self.weight)
        except Exception as e:
            logger.warning(f"Init-Task {self.fn.__name__} fehlgeschlagen: {e}")

class SplashScreen(QWidget):
    def __init__(self):
        super().__init__()
        logger.info("Showing splash screen")
        self.setWindowFlags(Qt.WindowType.FramelessWindowHint | Qt.WindowType.WindowStaysOnTopHint)
        self.setAttribute(Qt.WidgetAttribute.WA_TranslucentBackground)

        # UI-Aufbau
        self.frame = QFrame(self)
        self.frame.setFixedSize(500, 350)
        self.frame.setStyleSheet(f"""
            QFrame{{background:qlineargradient(x1:0,y1:0,x2:1,y2:1,
            stop:0 {STYLES['primary']},stop:1 {STYLES['secondary']});
            border-radius:20px;}}
        """)
        shadow = QGraphicsDropShadowEffect(self.frame)
        shadow.setBlurRadius(40)
        shadow.setOffset(0, 0)
        self.frame.setGraphicsEffect(shadow)

        layout = QVBoxLayout(self.frame)
        layout.setContentsMargins(40, 40, 40, 30)
        layout.setSpacing(10)
        layout.setAlignment(Qt.AlignmentFlag.AlignTop | Qt.AlignmentFlag.AlignHCenter)

        self.logo = QLabel()
        layout.addWidget(self.logo)

        name = QLabel("ShokaiShelf")
        name.setFont(QFont("Arial", 46, QFont.Weight.Bold))
        name.setStyleSheet(f"color:{STYLES['text_on_primary']};")
        name.setAlignment(Qt.AlignmentFlag.AlignCenter)
        layout.addWidget(name)
        layout.addSpacing(15)

        self.progress = QProgressBar()
        self.progress.setFixedHeight(8)
        self.progress.setRange(0, 100)
        self.progress.setValue(0)
        self.progress.setTextVisible(False)
        self.progress.setStyleSheet(f"""
            QProgressBar{{background:rgba(255,255,255,0.3);border-radius:4px}}
            QProgressBar::chunk{{background:{STYLES['text_on_primary']};border-radius:4px}}
        """)
        layout.addWidget(self.progress)

        self.state_label = QLabel("Checking for updates...")
        self.state_label.setFont(QFont("Arial", 10))
        self.state_label.setStyleSheet(f"color:{STYLES['text_on_primary']};")
        self.state_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        layout.addWidget(self.state_label)

        bottom = QHBoxLayout()
        ver = QLabel(f"v{__version__}")
        ver.setFont(QFont("Arial", 10))
        ver.setStyleSheet(f"color:{STYLES['text_on_primary']};opacity:0.7;")
        bottom.addWidget(ver, alignment=Qt.AlignmentFlag.AlignLeft)
        by = QLabel("Made by Chaptiv")
        by.setFont(QFont("Arial", 10))
        by.setStyleSheet(f"color:{STYLES['text_on_primary']};opacity:0.5;")
        bottom.addWidget(by, alignment=Qt.AlignmentFlag.AlignRight)
        layout.addLayout(bottom)

        self.show()
        self.center_on_active_screen()

        # Init-Signale
        self.init_signals = InitSignals()
        self.init_signals.progress.connect(self.on_init_progress)

        # Self-tests
        self.self_tests = [
            (self.init_database, (), 2),
            (self.load_settings, (), 1),
            (self.test_api, (), 3),
            (self.warm_cache, (), 2),
        ]
        self.total_units = sum(w for _fn, _args, w in self.self_tests)
        self.completed_units = 0

        # Starte Self-tests
        for fn, args, weight in self.self_tests:
            QThreadPool.globalInstance().start(InitWorker(fn, args, weight, self.init_signals))

    def center_on_active_screen(self):
        screen = QApplication.screenAt(QCursor.pos()) or QApplication.primaryScreen()
        geo = screen.availableGeometry()
        x = geo.x() + (geo.width() - self.frame.width()) // 2
        y = geo.y() + (geo.height() - self.frame.height()) // 2
        self.move(QPoint(x - self.frame.x(), y - self.frame.y()))

    # Self-test methods
    def init_database(self):
        get_database_connection()

    def load_settings(self):
        _ = settings.value("use_appstore_layout", False, bool)

    def test_api(self):
        try:
            resp = requests.head("https://graphql.anilist.co", timeout=5)
            resp.raise_for_status()
        except Exception as e:
            logger.warning(f"API-Test fehlgeschlagen: {e}")

    def warm_cache(self):
        load_pixmap("", (1, 1))

    # Progress handling
    def on_init_progress(self, weight: int):
        self.completed_units += weight
        pct = min(int(self.completed_units / self.total_units * 100), 100)
        self.progress.setValue(pct)
        self.state_label.setText(f"Performing self-tests… ({pct}%)")
        if pct >= 100:
            self.state_label.setText("Starting Program")
            QTimer.singleShot(1000, self.close_and_open_main)

    def close_and_open_main(self):
        self.close()
        try:
            self.main_window = AnimeManager()
            self.main_window.show()
        except Exception as e:
            logger.error(f"Failed to open main window: {e}")

# Main
def main():
    logger.info("Application start")
    app = QApplication(sys.argv)
    app.setStyle("Fusion")

    translator = QTranslator()
    locale = QLocale.system().name()
    if translator.load(f"translations/shokaishelf_{locale}.qm"):
        app.installTranslator(translator)

    if HAS_MATERIAL:
        apply_stylesheet(app, theme="light_purple.xml")

    # Updater ausführen
    from .updater import run_updater
    splash = SplashScreen()
    manifest = run_updater(__version__)
    if manifest:
        splash.state_label.setText(f"Update to {manifest['version']} installed, starting...")
        QTimer.singleShot(2000, splash.close_and_open_main)
    else:
        splash.state_label.setText("No updates found, starting...")

    sys.exit(app.exec())

if __name__ == "__main__":
    main()